//
//  IconChooser.swift
//  Do It
//
//  Created by Jim Dovey on 2/6/20.
//  Copyright © 2020 Jim Dovey. All rights reserved.
//

import SwiftUI

// START:IconChooser
struct IconChooser: View {
    @Binding var selectedIcon: String

    // END:IconChooser
    // START:SelectionInfo
    private struct IconSelectionInfo {
        let name: String
        let anchor: Anchor<CGRect>
    }

    private struct IconChoice: PreferenceKey {
        typealias Value = [IconSelectionInfo]
        static var defaultValue: Value = []
        static func reduce(value: inout [IconSelectionInfo],
                           nextValue: () -> [IconSelectionInfo]) {
            value.append(contentsOf: nextValue())
        }
    }
    // END:SelectionInfo

    // START:ButtonStyle
    private struct IconChoiceButtonStyle: ButtonStyle {
        func makeBody(configuration: Configuration) -> some View {
            configuration.label
                .font(.system(size: 24, weight: .bold, design: .rounded))
                .padding(6)
                .frame(width: 30)
                .padding(14)
                .background(Color(UIColor.tertiarySystemFill)) // <label id="code.ch5.uicolor.fill.tertiary" />
                .clipShape(Circle())
                .scaleEffect(configuration.isPressed ? 1.2 : 1)
        }
    }

    // START:IconChooser
    // START:ShowSelection
    var body: some View {
        VStack(spacing: 14) {
            // END:IconChooser
            // <literal:elide> ... </literal:elide>
            // END:ButtonStyle
            // END:ShowSelection
            // START:IconChooser
            ForEach(listIconChoices, id: \.self) { rowData in // <label id="code.ch5.icons.list-var" />
                HStack(spacing: 14) {
                    ForEach(rowData, id: \.self) { icon in
                        // START:AnchorPreference
                        Button(action: { self.selectedIcon = icon }) {
                            Image(systemName: icon)
                        }
                        // END:IconChooser
                        // START_HIGHLIGHT
                        .anchorPreference(key: IconChoice.self, value: .bounds) {
                            [IconSelectionInfo(name: icon, anchor: $0)]
                        }
                        // END_HIGHLIGHT
                        // END:AnchorPreference
                        // START:IconChooser
                    }
                }
            }
            // START:ButtonStyle
            // START:ShowSelection
        }
        // END:IconChooser
        .buttonStyle(IconChoiceButtonStyle())
        // END:ButtonStyle
        // START_HIGHLIGHT
        .backgroundPreferenceValue(IconChoice.self) { values in
            GeometryReader { proxy in
                self.selectionCircle(for: values, in: proxy)
            }
        }
        // END_HIGHLIGHT
        // START:ButtonStyle
        // START:IconChooser
    }
    // END:IconChooser
    // END:ShowSelection
    // END:ButtonStyle

    // START:DefineSelection
    private func selectionCircle(
        for prefs: [IconSelectionInfo],
        in proxy: GeometryProxy
    ) -> some View {
        let p = prefs.first { $0.name == selectedIcon } // <label id="code.ch5.icon.find-pref" />
        let bounds = p != nil ? proxy[p!.anchor] : .zero

        return Circle()
            .stroke(lineWidth: 3)
            .foregroundColor(Color(UIColor.separator)) // <label id="code.ch5.icon.select.color" />
            .frame(width: bounds.size.width + 12, // <label id="code.ch5.icon.select.padding" />
                   height: bounds.size.height + 12)
            .fixedSize()
            .offset(x: bounds.minX - 6, y: bounds.minY - 6)
    }
    // END:DefineSelection
}

struct IconChooser_Previews: PreviewProvider {
    static var previews: some View {
        // START:Preview
        StatefulPreviewWrapper("list.bullet") {
            IconChooser(selectedIcon: $0)
        }
        // END:Preview
    }
}
